package gov.va.mhv.util.xml;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import org.jdom2.Content;
import org.jdom2.Document;
import org.jdom2.Namespace;

/** Query an XML document for each of the XPath expressions given in selectMap.
 * The results are stored in the parallel resultMap.
 * @param <T>
 */
public class XPathMapSelector extends XPathSelectorAbstract
{
	protected final Logger logger = Logger.getLogger(getClass().getPackage().getName());

	protected Map<String,XPathSelectExpr> selectMap = null;

	public XPathMapSelector(Set<Namespace> namespaces, Map<String,XPathSelectExpr> selectMap) {
		super(namespaces);
		if (selectMap == null) {
			selectMap = new HashMap<String,XPathSelectExpr>();
		}
		this.selectMap = selectMap;
	}

	/**
	 * Pre-compile, or recompile the specifications in selectMap.
	 * This is optional. If not done, then expressions are compiled upon first encouter.
	 */
	private <T extends Content> void compileSelectors()
	{
		for (Map.Entry<String,XPathSelectExpr> sel : selectMap.entrySet()) {
			XPathSelectExpr<T> xse = sel.getValue();
			xse.xe = compileSelector(xse.expr, xse.filter);
		}
	}

	//--------------------------------------------------------------------------------

	/**
	 * For each selectMap item, find all matching elements.
	 * @param doc
	 * @return
	 */
	public <T extends Content> Map<String,List<T>> selectEachAll(Document doc)
	{
		Map<String,List<T>> resultMap = new HashMap<String,List<T>>();
		for (Map.Entry<String,XPathSelectExpr> sel : selectMap.entrySet()) {
			String name = sel.getKey();
			XPathSelectExpr<T> xse = sel.getValue();
			if (xse.xe == null) {
				xse.xe = compileSelector(xse.expr, xse.filter);
			}
			List<T> result = (List<T>) xse.xe.evaluate(doc);
			logger.fine("For " + descExpr(xse.xe) + ", found " + result.size() + " items");
			resultMap.put(name, result);
		}
		return resultMap;
	}

	/**
	 * For each selectMap item, find the first matching element.
	 * @param doc
	 * @return
	 */
	public <T extends Content> Map<String,T> selectEachFirst(Document doc)
	{
		Map<String,T> resultMap = new HashMap<String,T>();
		for (Map.Entry<String,XPathSelectExpr> sel : selectMap.entrySet()) {
			String name = sel.getKey();
			XPathSelectExpr<T> xse = sel.getValue();
			if (xse.xe == null) {
				xse.xe = compileSelector(xse.expr, xse.filter);
			}
			T result = (T) xse.xe.evaluateFirst(doc);
			logger.fine("For " + descExpr(xse.xe) + ", found: " + result);
			resultMap.put(name, result);
		}
		return resultMap;
	}


	// ================================================================================

	/**
	 * Transform the given result objects into their correpsonding String objects.
	 * @param resultMap
	 * @return
	 */
	public static Map<String,String> resultsAsStrings(Map<String,Content> resultMap)
	{
		Map<String,String> strMap = new HashMap<String,String>();
		for (Map.Entry<String,Content> item : resultMap.entrySet()) {
			Object resultItem = item.getValue();
			String resultValue = null;
			if (resultItem != null) {
				resultValue = ((Content)resultItem).getValue();
			}
			strMap.put(item.getKey(), resultValue);
		}
		return strMap;
	}
}
